%% Fourier Transformation
% Martin H. Trauth 10 June 2021
%
% Note: The following script uses i as the imaginary unit i = sqrt(-1). We
% should therefore not use it as a loop counter in a for loop. Furthermore,
% it is recommended to use 1i instead of just i for robustness.
%
% We first clear the workspace and the command window. We then close all
% figure windows.
clear, clc, close all

%%
% We then define the noise level, i.e. the standard deviation of the
% Gaussian noise, the sampling frequency Fs and create a time vector t.
Ns = 2;
Fs = 2;
t = 1/Fs : 1/Fs : 500; t = t';

%%
% According to Euler's formula y=exp(1i*t) is equal to y =cos(t)+1i*sin(t).
% Both cos(t) and sin(t) have a period of 2*pi. Therefore cos(2*pi*t/T) and
% sin(2*pi*t/T) have a period of T. The corresponding frequency is f = 1/T.
% We use Euler's formula to generate a signal with a period of five.
x = exp(1i*2*pi*t/5);
x = real(x);

figure('Position',[100 800 800 400],...
    'Color',[1 1 1])
axes('XLim',[0 100],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(t,x,...
    'LineWidth',1)

%%
% We then create a composite signal of multiple sine waves with different
% periods T, different amplitudes A and without noise using Euler's
% formula.
T = [50 15 5];
A = [2 1 0.5];

for j = 1 : length(T)
    xj(:,j) = A(j)*exp(1i*2*pi*t/T(j));
end
x = real(sum(xj,2));

figure('Position',[100 700 800 400],...
    'Color',[1 1 1])
axes('XLim',[0 100],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(t,x,...
    'LineWidth',1)

%%
% We then create a composite signal of multiple sine waves with different
% periods T, different amplitudes A and without noise using trigonometric
% functions.
x =   2*sin(2*pi*t/50) + ...
        sin(2*pi*t/15) + ...
    0.5*sin(2*pi*t/5);

figure('Position',[100 600 800 400],...
    'Color',[1 1 1])
axes('XLim',[0 100],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(t,x,...
    'LineWidth',1)

%%
% We can use the Fast Fourier Transformation to calculate a Fourier
% spectrum of the signal, displayed as the power spectral density against
% the frequency f and the period T.
nfft = 2^nextpow2(length(t));
Xxx = fft(x,nfft);
Pxx2 = abs(Xxx).^2 /Fs /length(x);
Pxx = [Pxx2(1); 2*Pxx2(2:nfft/2)];
f = 0 : Fs/(nfft-1) : Fs/2; f = f';

figure('Position',[100 500 800 400],...
    'Color',[1 1 1])
axes('Position',[0.1 0.1 0.35 0.8],...
    'XLim',[0 Fs/2],...
    'YLim',[0 max(Pxx)],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(f,Pxx,...
    'LineWidth',1)
xlabel('Frequency')
ylabel('Power Spectral Density')
axes('Position',[0.55 0.1 0.35 0.8],...
    'XLim',[2/Fs 70],...
    'YLim',[0 max(Pxx)],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(1./f,Pxx,...
    'LineWidth',1)
xlabel('Period')
ylabel('Power Spectral Density')

%%
% We then create a composite signal of multiple sine waves with different
% periods T, different amplitudes A and additive Gaussian noise Ns.
rng(0)
xn =  2*sin(2*pi*t/50) + ...
        sin(2*pi*t/15) + ...
    0.5*sin(2*pi*t/5) + ...
     Ns*randn(size(t));

figure('Position',[100 400 800 400],...
    'Color',[1 1 1])
axes('XLim',[0 100],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(t,xn,...
    'LineWidth',1)

%%
% We can use the Fast Fourier Transformation to calculate a Fourier
% spectrum of the signal, displayed as the power spectral density against
% the frequency f and the period T.
nfft = 2^nextpow2(length(t));
Xxx = fft(xn,nfft);
Pxx2 = abs(Xxx).^2 /Fs /length(xn);
Pxx = [Pxx2(1); 2*Pxx2(2:nfft/2)];
f = 0 : Fs/(nfft-1) : Fs/2;  f = f';

figure('Position',[100 300 800 400],...
    'Color',[1 1 1])
axes('Position',[0.1 0.1 0.35 0.8],...
    'XLim',[0 Fs/2],...
    'YLim',[0 max(Pxx)],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(f,Pxx,...
    'LineWidth',1)
xlabel('Frequency')
ylabel('Power Spectral Density')
axes('Position',[0.55 0.1 0.35 0.8],...
    'XLim',[2/Fs 70],...
    'YLim',[0 max(Pxx)],...
    'XGrid','On',...
    'YGrid','On'), hold on
line(1./f,Pxx,...
    'LineWidth',1)
xlabel('Period')
ylabel('Power Spectral Density')

